#!/usr/bin/env perl
# HSV Parser Tests (Perl)

use strict;
use warnings;
use FindBin;
use lib $FindBin::Bin;
use HSV qw(parse SOH STX ETX SO SI DLE FS GS RS US);
use Data::Dumper;

print "=" x 50, "\n";
print "HSV Parser Tests (Perl)\n";
print "=" x 50, "\n";

my $passed = 0;
my $failed = 0;

sub test {
    my ($name, $code) = @_;
    eval { $code->(); };
    if ($@) {
        print "✗ $name: $@";
        $failed++;
    } else {
        print "✓ $name\n";
        $passed++;
    }
}

sub assert_eq {
    my ($got, $expected, $msg) = @_;
    $msg //= '';
    if (ref($got) eq 'ARRAY' && ref($expected) eq 'ARRAY') {
        die "$msg: arrays differ\n" unless @$got == @$expected;
        for my $i (0..$#$got) {
            die "$msg: element $i differs\n" unless $got->[$i] eq $expected->[$i];
        }
    } elsif (!ref($got) && !ref($expected)) {
        die "$msg: got '$got', expected '$expected'\n" unless $got eq $expected;
    } else {
        die "$msg: type mismatch\n";
    }
}

test('Basic parsing', sub {
    my $result = parse(STX . "name" . US . "Alice" . RS . "age" . US . "30" . ETX);
    assert_eq(scalar(@{$result->{records}}), 1);
    assert_eq($result->{records}[0]{name}, 'Alice');
    assert_eq($result->{records}[0]{age}, '30');
});

test('Multiple records', sub {
    my $result = parse(STX . "name" . US . "Alice" . FS . "name" . US . "Bob" . ETX);
    assert_eq(scalar(@{$result->{records}}), 2);
});

test('Array values', sub {
    my $result = parse(STX . "tags" . US . "a" . GS . "b" . GS . "c" . ETX);
    assert_eq($result->{records}[0]{tags}, ['a', 'b', 'c']);
});

test('SOH header', sub {
    my $result = parse(SOH . "hsv" . US . "1.0" . RS . "type" . US . "users" . STX . "name" . US . "Alice" . ETX);
    assert_eq($result->{header}{hsv}, '1.0');
    assert_eq($result->{header}{type}, 'users');
    assert_eq(scalar(@{$result->{records}}), 1);
});

test('SO/SI nesting', sub {
    my $result = parse(STX . "user" . US . SO . "name" . US . "Alice" . RS . "email" . US . 'a@b.com' . SI . ETX);
    assert_eq($result->{records}[0]{user}{name}, 'Alice');
    assert_eq($result->{records}[0]{user}{email}, 'a@b.com');
});

test('Deep nesting', sub {
    my $result = parse(STX . "data" . US . SO . "level1" . US . SO . "level2" . US . "deep" . SI . SI . ETX);
    assert_eq($result->{records}[0]{data}{level1}{level2}, 'deep');
});

test('Newlines in values', sub {
    my $result = parse(STX . "text" . US . "line1\nline2\nline3" . ETX);
    assert_eq($result->{records}[0]{text}, "line1\nline2\nline3");
});

test('Quotes (no escaping)', sub {
    my $result = parse(STX . "msg" . US . 'He said "hello"' . ETX);
    assert_eq($result->{records}[0]{msg}, 'He said "hello"');
});

test('Mixed content', sub {
    my $result = parse("ignored" . STX . "name" . US . "Alice" . ETX . "also ignored");
    assert_eq(scalar(@{$result->{records}}), 1);
    assert_eq($result->{records}[0]{name}, 'Alice');
});

test('Multiple blocks', sub {
    my $result = parse(STX . "a" . US . "1" . ETX . "junk" . STX . "b" . US . "2" . ETX);
    assert_eq(scalar(@{$result->{records}}), 2);
});

test('Nested structure with array', sub {
    my $result = parse(STX . "user" . US . SO . "name" . US . "Alice" . RS . "tags" . US . "admin" . GS . "user" . SI . ETX);
    assert_eq($result->{records}[0]{user}{name}, 'Alice');
    assert_eq($result->{records}[0]{user}{tags}, ['admin', 'user']);
});

print "=" x 50, "\n";
print "$passed passed, $failed failed\n";
print "=" x 50, "\n";

exit($failed > 0 ? 1 : 0);
